;   This file is software for PIC16F88 based automatic logging		*
;	weather station. It reads temperature and rainfall hourly		*
;	and flashes a LED briefly every 4 seconds to indicate normal	*
;	operation. Rain sensor triggers interupt and increments a		*
;	counter that is reset every hour. Rain value is written to a	*
;	separate EEPROM each hour.										*
;																	*
;	Rain sensor is a tipping bucket with single switch (magnetic	*
; 	reed) switch closure every bucket tip.  Same switch closes		*
;	momentarily each bucket tip (regardless of direction of tip).	*
;																	*
;   Temperature is also read each hour and two bytes (10 bits)		*
;	is written to EEPROM.  Temperature is sensed with a DS1621		*
;	sensor that uses the I2C bus protocol.							*
;																	*
;   All peripherals shut down and processor sent to sleep when no	*
;	interupt is being processed. Clock oscillator is 32 KHz xtal	*
;	on TMR1.  Code clock is internal oscillator running at 250 KHz.	*
;																	*
;	Current location in EEPROM that is written to is based on hour	*
;	counter. This counter must be manually reset when the EEPROM is	*
;	swapped. A switch is provided which must be held closed at the	*
;	time the LED flashes (every 4 secs). If the hour counter reset	*
;	occurs successfully, the LED will quickly flash 3 times.		*
;                                                                     *
;**********************************************************************
;                                                                     *
;    Filename:	    Weather station.asm                               *
;    Date:          27 Sept 2006                                      *
;    File Version:  2                                                 *
;                                                                     *
;    Author:        Glenn Pure                                        *
;    Company:                                                         *
;                                                                     *
;                                                                     *
;**********************************************************************
;                                                                     *
;    Files required:                                                  *
;                                                                     *
;                                                                     *
;                                                                     *
;**********************************************************************
;                                                                     *
;    Notes: Automatic weather station, version 2                      *
;           (Version 2 uses a DS1621 temp sensor                      *
;           and a magnetic reed switch rain sensor)                   *
;                                                                     *
;                                                                     *
;**********************************************************************

	list      p=16f88           ; list directive to define processor
	#include <p16F88.inc>        ; processor specific variable definitions

	errorlevel  -302              ; suppress message 302 from list file

; Code protect off, CCP1 on RB0, debug off, write protect all code space, data (EERAM) code protect off, low V programming off, brown out detector off,
; MCLR internally set to Vdd, power up timer on, watchdog timer off, internal oscillator with RA6 and RA7 as I/O (need to set OSCON to control frequency)
	__CONFIG    _CONFIG1, _CP_OFF & _CCP1_RB0 & _DEBUG_OFF & _WRT_PROTECT_ALL & _CPD_OFF & _LVP_OFF & _BODEN_OFF & _MCLR_OFF & _PWRTE_ON & _WDT_OFF & _INTRC_IO
; Internal-external clock switch disabled, fail safe clock monitor off
	__CONFIG    _CONFIG2, _IESO_OFF & _FCMEN_OFF

; '__CONFIG' directive is used to embed configuration word within .asm file. 
; The labels following the directive are located in the respective .inc file.
; See data sheet for additional information on configuration word settings.


; Variables common to all RAM banks (total of 16 bytes)
ALLBANK		UDATA_SHR	0x70	
secs_4		RES 1			; incremented every 4 seconds
mins		RES 1			; minutes elapsed in hour
rec_num_l	RES 1			; record address counter (low byte)
rec_num_h	RES 1			; record address counter (high byte)
rainfall	RES 1			; rain bucket tip counter
t_flag		RES 1			; rainfall switch status (RA6 or RA7 high currently?)			
temp_l		RES 1			; temperature low byte
temp_h		RES 1			; temperature high byte
aa			RES 1			; general purpose registers
bb			RES 1
cc			RES 1			; used in serial eeprom writing routine to hold data
dd			RES 1			; used in serial eeprom writing routine for bit count
ee			RES 1			; used as sequence counter for serial eeprom writes
ff			RES 1
gg			RES 1			; flag for error communicating with DS1621 sensor
hh			RES 1


; Bank specific uninitialized Data Section (total of 80 bytes)
; Note, BANK0 20h onwards also maps to BANK2
; These RAM locations mostly used to store data read from EEPROM and processed ready for tranmission
; over serial port to computer
BANK0		UDATA	0x20	; explicit address specified is not required
addr_low	RES 1			; low byte of 24C eeprom for reading
addr_hi		RES 1			; high byte of 24C eeprom
loop_i		RES 1			; inner loop counter for reading eeproms
loop_o		RES 1			; outer loop counter for reading eeproms
set_no		RES 1			; data set number read from eeprom (used to detect reboots during logging)
rd01		RES 1			; rd01 to rd4 are raw data values from EEPROM
rd02		RES 1
rd03		RES 1
rd04		RES 1
dig1		RES 1			; dig1 to dig5 are for ascii data to transmit
dig2		RES 1
dig3		RES 1
dig4		RES 1
dig5		RES 1
clk_adj		RES 1			; flag to indicate whether clock adjust routine is active.
elap24		RES 1			; counter to measure whether 24 hrs elapsed

;***** EERAM addresses
RECNUM_L		EQU	0x00		; Elapsed records x 4 (low byte)
RECNUM_H		EQU 0x01		; Elapsed records x 4 (high byte)
FIX_CLK1		EQU 0x02		; Clock error value of secs_4 to reload every 24 hrs
FIX_CLK2		EQU	0x03		; Clock error TMR1H value to reload every 24 hrs


;***** Initialise EERAM data
	ORG	0x2100	; start at EERAM location 00
	DE	0x00
	DE	0x00
	DE	D'15'	; This and next value assures clock correction is almost zero
	DE	0xff

;***** Various defined constants
LOG_FREQ	EQU D'30'		; Frequency (minutes) with which data is logged
HRS24		EQU D'48'		; = (No of minutes in 24 hours)/LOG_FREQ
; Change the above two values if you want to change the logging frequency (max value 255 for either)
; ie, most frequent logging interval is 6 minutes (if normal 24 hr clock used)
SCL			EQU D'3'		; PORTB SCL pin bit number
SDA			EQU D'1'		; PORTB SDA pin bit number
T_ERR		EQU D'5'		; temperature reading error (added to actual temp to correct)
DOT			EQU 0x2E		; ASCII "."
COMMA		EQU 0x2C		; ASCII ","
CR			EQU	0x0D		; ASCII carriage return
LF			EQU	0x0A		; ASCII line feed






;**********************************************************************
RST		CODE     0x000			; processor reset vector
	goto    main				; go to beginning of program
	

INTV	CODE     0x004			; INTERRUPT ROUTINE
	banksel	PIR1
	btfsc	PIR1,TMR1IF			; Check if TMR1 interupt occurred
	goto	t_int				; Yes - jump to handle
	btfsc	INTCON,INTF			; No. Check if RB0 interupt occurred (INTCON in all ram pages)
	goto	rain				; Yes - jump to handle: rain sensor incremented
	btfsc	INTCON,RBIF			; PortB4:7 change interupt?
	goto	clk_err				; yes: jump to process
	goto	main				; No valid interupt: suspect: reboot

;**********************************************************
t_int	; real time clock routine
	clrf	PIR1				; first clear interupt flag
	decf	secs_4,1			; update 4 second counter
	btfsc	PORTA,2				; check if record counter reset switch closed
								; (user must hold this switch closed during a clock interupt or it will be ignored)
	call	rec_clr				; zero counters and exit interupt (if confirmed closure)
	btfsc	PORTA,0				; otherwise, check if 'download data' switch closed
	call	dump				; jump if so (premature return will occur if error communication with eeproms)
	banksel	PORTA				; ensure correct bank following returns
	bsf		PORTA,4				; otherwise, continue clock routine: flash LED briefly (every 4 secs)
	movlw	D'64'				; time loop counter
	movwf	aa
; waste time (about 3 msec with 250 KHz clock)
	decfsz	aa,1
	goto	$-1					; loop until aa is zero
	bcf		PORTA,4				; switch led off
	movf	t_flag,1			; Is flag set to read temp and rainfall and write to serial eeprom?
	btfsc	STATUS,Z
	goto	more_clk			; No: continue normal clock routine
	decfsz	t_flag,1			; Yes: check value in t_flag to see if it has decremented to zero
	goto	more_clk			; No: continue normal clock
	goto	data_wr				; Yes: jump to read temp and write data routine
								; Note, "call" not needed here as write never occurs on minute or hour boundary
more_clk						; so rest of clock code can be ignored
	btfss	clk_adj,3			; Is clock error routine active? (check every 4 secs)
	goto	norm_clk			; No: continue normal clock routine
	decfsz	elap24,0			; Yes: In last LOG_FREQ interval before 24 hrs elapsed?
	goto	norm_clk			; No: continue normal clock routine
	decfsz	mins,0				; Yes: in last minute?
	goto	norm_clk			; No: continue normal clock routine
	movlw	D'5'				; In last 20 secs before 24 hrs elapsed?
	subwf	secs_4,0
	btfss	STATUS,Z
	goto	norm_clk			; No: continue normal clock routine
	bsf		PORTA,4				; Yes: turn LED on until user presses clock correction button (PORTB:4)
	goto	err_poll			; Jump to routine to poll for switch closure and keep track of clock increments
norm_clk
	movf	secs_4,1			; Test if secs_4 is zero by writing to itself
	btfss	STATUS,Z			; if zero, 60 seconds has elapsed
	retfie						; no: return from interrupt (instruction sets GIE)
	movlw	D'15'				; yes: reload 4 second counter
	movwf	secs_4
	decf	mins,1				; next, update minutes
	btfss	STATUS,Z			; has logging frequency (mins) elapsed?
	retfie						; no: return from interrupt (instruction sets GIE)
	movlw	LOG_FREQ			; yes: reload minute counter
	movwf	mins
	bcf		INTCON,RBIE			; Ensure RB7:4 port change interupt disabled from first LOG_FREQ interval onwards
								; (if user wants to adjust clock error, must do so within LOG_FREQ minutes of a reset)
	bcf		INTCON,RBIF			; Ensure RB7:4 port change interupt flag also clear
	movlw	D'5'
	movwf	bb					; set up counter for call to t_log (in case error on first try)
	btfss	gg,4				; check if error communicating with DS1621 after last communication
	goto	$+3					; no: jump for normal processing
	call	init1621			; yes: attempt to initialise, then wait (to allow initialisation to complete)
	call	wait_20
	nop
	call	t_log				; Call routine to start temperature read on DS1621 (takes 1 second to record)
	addlw	D'1'				; did return occur with W containing 0xff? (ie no error)
	btfsc	STATUS,Z
	goto	tlog_ok				; no error: finish up
	decfsz	bb,1				; or check loop repeat counter and repeat
	goto	$-5					; if necessary (if fail after five repeats, just continue)
tlog_ok
	movlw	D'2'				; then set flag so that temp is read from DS1621 after two more 4 sec interupts
	movwf	t_flag				; (need to do this after two more to cover situation at 24 hrs where TMR1H may
								; be loaded with a value that causes next timer interupt in less than the 1 sec
								; needed for the temperature to be recorded by the DS1621)
	banksel	PORTB				; ensure in page 0
	decfsz	elap24,1			; determine if 24 hr counter has decremented to zero
	retfie						; no: finished: return from interrupt (instruction sets GIE)
	movlw	HRS24				; yes: reload 24 hour counter
	movwf	elap24
	banksel	EEADR				; and perform the clock correction
	movlw	FIX_CLK1			; Fetch secs_4 correction value
	movwf	EEADR
	banksel	EECON1
	bcf		EECON1, EEPGD
	bsf		EECON1, RD
	banksel	EEDATA
	movf	EEDATA,0
	movwf	secs_4				; and copy the correction value to secs_4
	incf	EEADR,1				; point to eeram address for TMR1H correction
	banksel	EECON1
	bcf		EECON1, EEPGD
	bsf		EECON1, RD
	banksel	EEDATA
	movf	EEDATA,0
	banksel	T1CON				; Page 0
	bcf		T1CON,TMR1ON		; avoid write collision while timer running by turning off first
	movwf	TMR1H
	bsf		T1CON,TMR1ON		; turn timer back on
	retfie						; and finish

;******************************************************************************
data_wr		; Note, this code only executed once per logging interval, 8 seconds after interval
			; has incremented. Jump is from clock routine.
			; Clock code above asks DS1621 to read temp 8 seconds before.
			; Code here reads the temperature from the DS1621 and writes to EEPROM
			; It also writes rainfall to EEPROM then clears rainfall counter.
			; Next, record counters (high and low) in eeram of PIC are updated.
			; Finally, it waits for the EEPROM write to finish before writing 0xff to the
			; next 4 bytes - this will mark last data written in a way that can be detected
			; by dumping the EEPROMs. Useful if a problem occurs.
	movlw	D'5'
	movwf	bb					; set up counter for temperature read
	btfss	gg,4				; check if error communicating with DS1621 after last communication
	goto	t_rpt				; no: jump for normal processing
	call	init1621			; yes: attempt to initialise
	goto	t_error				; don't attempt to read temp now since DS1621 just reset (at best)
t_rpt
	call	t_read				; read the temperature and check for error
	addlw	D'1'				; did return occur with W containing 0xff? (ie no error)
	btfsc	STATUS,Z
	goto	do_ee				; no error: write the results to EEPROM
	decfsz	bb,1				; error: check loop repeat counter and repeat reading if loop not zero
	goto	t_rpt				; if fail after five repeats, put dummy data in temp_h and temp_l
t_error
	clrf	temp_h				; place zero in temp_h and temp_l as signal to user
	clrf	temp_l
do_ee	
	call	read_cnt			; first, make sure ram values of record counter (low and high) are OK
	movlw	D'5'
	movwf	bb					; set up counter for write attempts (in case error on first write)
wr_data
	call	write24c			; call routine to write temp and rainfall to serial eeprom
	addlw	D'1'				; did return occur with W containing 0xff? (ie no error)
	btfsc	STATUS,Z
	goto	wr_fin				; no error: finish up
	decfsz	bb,1				; error: check write loop repeat counter and repeat write
	goto	wr_data				; if necessary (if fail after five repeats, just continue)
								; EEPROM will contain unwritten 4 bytes at this location
wr_fin
; after data is written to eeprom, increment eeprom address pointers and data set counter.
; Code moved from clock routine to here so that increment occurs after write (not before) so if address counter is reset,
; first set of data will be written starting at address 00:00
	incf	ee,1				; first increment data set counter (resets to zero if a reboot occurs)
	movlw	D'4'				; Update record counter (low)
	addwf	rec_num_l,1			; increments 4 every data write so that the rec_num_l:rec_num_h pair
								; map directly to the start address in the EEPROM to be written next
	btfss	STATUS,C			; rec_num_l counter rolled over?
	goto	wr_addr				; no: clean up then return from interrupt
	incf	rec_num_h,1			; yes: increment record counter high byte (no need to reset if it rolls over
								; since it never should)

;	finally, write new record count (high and low) to EERAM in PIC
wr_addr
	banksel	EEADR				; set up for write (low byte first)
	movlw	RECNUM_L
	movwf	EEADR				; set up write address
	movf	rec_num_l,0			; select data to write
	movwf	EEDATA
	call	ee_write			; write the results (returns in page 0)
	banksel	EEADR				; set up for write (high byte now)
	incf	EEADR,1				; set up write address
	movf	rec_num_h,0			; select data to write
	movwf	EEDATA
	call	ee_write			; write the results (returns in page 0)
;	this section writes 0xff to the next 4 bytes (first waits for write of previous bytes to be completed
;	by calling write24c which will return with an error if the write operation is still underway.  This is
; 	because the EEPROMs will not ACK any attempt to communicate while the write is still underway)
	movf	ee,0				; Preserve value of data set counter before writing FF to it
	movwf	hh					; (use hh register which is only used by t_read routine)
	movlw	0xff				; first, set data set counter, rainfall, temp_l and temp_h to 0xff
	movwf	ee
	movwf	rainfall
	movwf	temp_h
	movwf	temp_l
	movlw	D'40'
	movwf	bb					; set up counter for write attempts - need to poll for long enough
								; until last 4 bytes written (about 5 msec, or ~ 300 instruction cycles at
								; 250 KHz clock (= 62 KHz instruction frequency)).  Loop to call write24c
								; takes about 35 instructions (including first portion of write24c to point
								; where first sendbyte() fails.  Hence number of repeats should be at least
								; 11: set to 40 to be safe.
wr_nxt
	call	write24c			; call routine to write temp and rainfall to serial eeprom
	addlw	D'1'				; did return occur with W containing 0xff? (ie no error)
	btfsc	STATUS,Z			; only get error if EEPROMs still writing previous 4 bytes of data.
	goto	$+3					; no error: finish up
	decfsz	bb,1				; error: check write loop repeat counter and repeat write
	goto	wr_nxt				; if necessary (if fail after bb = 0, just continue)
;	now quit
	clrf	rainfall			; but first set rainfall counter to zero
	movf	hh,0				; restore value of data set counter
	movwf	ee
	retfie						; return from interrupt (instruction sets GIE)
	
;***********************************************************************
rec_clr		; routine to clear record number counters (if requested by user): jump here from clock routine
	clrf	aa					; confirm switch closure
	btfss	PORTA,2				; check if counter reset switch closed still
	return						; if not, keep processing clock routine
	decfsz	aa,1				; waste time (about 1/100 sec with 256 KHz clock)
	goto	$-3					; loop counter not yet zero - keep looping
;	OK, switch closure confirmed: clear record counters
	banksel	EEADR				; set up for write (low byte first)
	movlw	RECNUM_L
	movwf	EEADR				; set up write address
	clrf	EEDATA				; zero low count and write
	call	ee_write			; write the results (returns in page 0)
	banksel	EEADR				; set up for write (high byte now)
	incf	EEADR,1				; set up write address
	clrf	EEDATA				; clear high count and write
	call	ee_write			; write the results (returns in page 0)
	clrf	rec_num_h			; next, update ram values of hours (low and high)
	clrf	rec_num_l

	; finally, flash LED briefly three times in a row to signal success, then exit
	banksel	PORTA
	bsf		PORTA,4
	clrf	aa					; loop counter
	decfsz	aa,1				; waste time (about 1/100 sec with 256 KHz clock)
	goto	$-1					; loop counter not yet zero - keep looping
	bcf		PORTA,4				; switch led off and leave it off for about 1/25 sec
	movlw	D'5'				; set outer loop counter
	movwf	bb
	clrf	aa					; inner loop counter
	nop
	nop
	decfsz	aa,1				; waste time (about 1/100 sec with 256 KHz clock)
	goto	$-3					; loop counter not yet zero - keep looping
	decfsz	bb,1				; outer loop
	goto	$-6
	bsf		PORTA,4				; flash LED briefly three times in a row to signal success
	clrf	aa					; loop counter
	decfsz	aa,1				; waste time (about 1/100 sec with 256 KHz clock)
	goto	$-1					; loop counter not yet zero - keep looping
	bcf		PORTA,4				; switch led off and leave it off for about 1/40 sec
	movlw	D'5'				; set outer loop counter
	movwf	bb
	clrf	aa					; inner loop counter
	nop
	nop
	decfsz	aa,1				; waste time (about 1/100 sec with 256 KHz clock)
	goto	$-3					; loop counter not yet zero - keep looping
	decfsz	bb,1				; outer loop
	goto	$-6
	bsf		PORTA,4				; flash LED briefly three times in a row to signal success
	clrf	aa					; loop counter
	decfsz	aa,1				; waste time (about 1/100 sec with 256 KHz clock)
	goto	$-1					; loop counter not yet zero - keep looping
	bcf		PORTA,4				; switch led off and quit
	goto	main				; done: reboot (to reset seconds and minute counters etc)

;***************************************************************
rain		; Routine to increment rainfall counter
			; Note that since RB0 is edge triggered, repeat interupts shouldn't occur
			; from the same bucket tip if debouncing of closure done properly
			; Note that switch opening at end is not debounced.  This should not cause a problem
			; since the closure is debounced at the start and any noisy switch opening at the end
			; should therefore not trigger another rainfall increment.
	bcf		INTCON,INTF			; first clear interupt flag
	call	wait_20				; check if B0 still closed after 20 msec
	banksel	PORTB
	btfss	PORTB,0
	goto	inc_rain			; RB0 switch confirmed closed: jump to process
	retfie						; RB0 closure not debounced: spurious interupt: exit
inc_rain
	incf	rainfall,1			; if rain sensor switch confirmed closed, increment rainfall counter
	retfie						; and exit
	

;***************************************************************
clk_err		; Routine to record clock error over 24 hrs and store correction value for ongoing use
			; Routine entered from RB7:4 port change interupt, which is disabled after LOG_FREQ minutes from power up
			; or reset. Hence, user must generate this interupt before then to perform the clock correction.
			; Uses value in elap24 to determine if 24 hrs has passed.  This does not have to be initialised
			; since the routine can only be entered before this variable is first decremented after a reset.
	banksel	PORTB
	bcf		INTCON,RBIF			; First clear interupt flag
	bcf		INTCON,RBIE			; and disable the port change interupt
	btfss	PORTB,4				; Test that RB4 was closed (clock error routine)
	goto	main				; No: suspicious: reboot
	btfsc	clk_adj,3			; Yes: check if this routine already started?
	goto	calc_err			; Yes: must be 24 hrs elapsed: jump to calculate and record error
	bsf		clk_adj,3			; No, continue: Set flag to indicate this routine is active
	movlw	D'15'
	movwf	secs_4				; reset second and minute counters
	movlw	LOG_FREQ
	movwf	mins
	clrf	TMR1L				; Reset timer 1
	clrf	TMR1H				; 
	clrf	PIR1				; clear TMR1 interupt flag (in case it is set)
	bsf		PORTA,4				; Turn LED on until next clock interupt (4 seconds) as a signal to user
	retfie						; Done: exit
calc_err	; jump here to calculate the clock error after 24hrs
	clrf	clk_adj				; first clear flag to indicate this routine is now done
	movf	TMR1H,0				; then read value of Timer1 high byte (each unit = 1/64 sec; no need to read low byte)
	movwf	bb					; store value in temporary register
	comf	bb,1				; negate the value (approximately)
	decf	elap24,0			; Next, check for valid value in elap24 (has to be HRS24 or 1)
	btfsc	STATUS,Z
	goto	clk_slow			; elap24 was 1: clock is slow
	movlw	HRS24				; Not 1: was it = HRS24?
	subwf	elap24,0
	btfss	STATUS,Z
	goto	main				; not = 1 or HRS24: suspicious: reboot
; This portion of routine executed when clock is fast
	movlw	LOG_FREQ			; minutes counter should be equal to LOG_FREQ, as rollover and reset of mins has just occurred
	subwf	mins,1				; Clock shouldn't be out by more than a few seconds in 24 hrs
	btfss	STATUS,Z
	goto	main				; Mins not equal to LOG_FREQ: suspicious: reboot
	movlw	D'4'				; Mins equal to LOG_FREQ. Now determine value of 4 second counter
	movwf	cc					; loop counter
	movlw	D'15'
	movwf	aa					; clock (secs_4) correction value
	incf	aa,1
	subwf	secs_4,0			; is value in secs_4 = 15?
chk_fast
	btfsc	STATUS,Z
	goto	wr_clker			; yes, match with secs_4 value: write error values and finish
	decfsz	cc,1				; no: loop done four repeats?
	goto	$+2					; no: keep looping
	goto	main				; loop done: clock out by more than 20 secs in 24 hrs: unlikely: reboot
	incf	aa,1				; no: keep track of secs_4 correction value
	addlw	D'1'				; then increment (secs_4 - 15) result and test again
	goto	chk_fast
	
clk_slow	; this portion of routine executed when clock is slow
	decfsz	mins,1				; Clock shouldn't be out by more than a few seconds in 24 hrs, ie, mins = 1
								; (clock in final 20 sec before rollover of 24 hrs)
	goto	main				; Mins not equal to 1: suspicious: reboot
	movlw	D'4'				; Mins equal to 1. Now determine value of 4 second counter
	movwf	cc					; loop counter
	movlw	D'15'
	movwf	aa					; clock (secs_4) correction value
chk_slow
	decf	secs_4,1			; decrement value in secs_4 to test contents
	btfsc	STATUS,Z
	goto	wr_clker			; reached zero, write error value and finish
	decfsz	cc,1				; not yet zero: loop done four repeats?
	goto	$+2					; no: keep looping
	goto	main				; loop done: clock out by more than 20 secs in 24 hrs: unlikely: reboot
	decf	aa,1				; otherwise, keep track of secs_4 correction value
	goto	chk_slow			; and keep looping

wr_clker	; write the clock error results to eeprom and finish
	banksel	EEADR				; set up for write (low byte first)
	movlw	FIX_CLK1
	movwf	EEADR				; set up write address
	movf	aa,0
	movwf	EEDATA				; place secs_4 adjustment ready for write
	call	ee_write			; write the results (returns in page 0)
	banksel	EEADR				; set up for write (high byte now)
	incf	EEADR,1				; set up write address
	movf	bb,0
	movwf	EEDATA				; place TMR1H adjustment ready for write
	call	ee_write			; write the results (returns in page 0)
	goto	no_rb47				; Done: reboot (will switch off LED also)
	

;***************************************************************
err_poll	; Routine to poll PORTB:4 switch closure for clock error correction routine
			; (jumps here from main clock routine)
			; This routine also has to poll the TMR1 interupt to keep the clock counters updated
			; Loop runs continuously after user presses RB4 switch.  However, if this occurs more
			; than ~20 sec into the next 24 hr period, the calc_err routine will cause a reboot
			; and no error values will be recorded.
	banksel	PORTB
	btfsc	PORTB,4				; Has user pressed the error correction switch?
	goto	calc_err			; Yes: jump to record error and reboot
	btfss	PIR1,TMR1IF			; RB4 not pressed: did timer1 interupt occur? (need to decrement secs_4 if so)
	goto	wait_sw				; No: waste time then loop again for switch closure or TMR1 interupt
	bcf		PIR1,TMR1IF			; Yes: first clear interupt flag
	decfsz	secs_4,1			; then decrement secs_4 and test if zero
	goto	wait_sw				; Not zero: waste time then loop again for switch closure or TMR1 interupt
	movlw	D'15'				; secs_4 now zero: reload 4 second counter
	movwf	secs_4
	decfsz	mins,1				; next, update minutes and check if zero
	goto	wait_sw				; Not zero: waste time then loop again for switch closure or TMR1 interupt
	movlw	LOG_FREQ			; yes: reload minute counter
	movwf	mins
	decfsz	elap24,1			; and decrement 24 hr counter and check if zero
	goto	wait_sw				; Not zero: waste time then loop again for switch closure or TMR1 interupt
	movlw	HRS24				; yes: reload 24 hour counter
	movwf	elap24
wait_sw
	movlw	D'10'				; Has 20 seconds elapsed since rollover of 24 hours?
	subwf	secs_4,0			; ie user failed to press clock error switch when they should have
	btfsc	STATUS,Z
	goto	main				; Yes: User forgot to press switch: REBOOT (reboot will also switch off LED)
	call	wait_20				; No: waste time then loop again to check for switch closure or interupt
	goto	err_poll			; or TMR1 interupt


;*******************************************************************
CONT	CODE
main
;	Initialise bank 0 items
	banksel	ADCON0
	movlw	B'00011000'			; GIE off, TMR0 interupt off, RB0 interupt on, RB7:4 change on, peripheral interupts off, flags 0 (for now)
	movwf	INTCON				; INTCON actually occurs in all 4 pages
no_rb47							; jump location for clock error routine to avoid enabling RB7:4 port change interupt
	movlw	B'11000000'			; Set up ADCON0
	movwf	ADCON0				; Use internal RC clock for conversion, RA0 input, A/D fetch off, A/D module off
	movlw	B'00011111'			; TMR1: don't use as system clock, prescaler 1:2, external crystal clock with driver on, don't sync to internal clock, timer on
	movwf	T1CON				
	movlw	B'00000000'
	movwf	T2CON				; Timer 2 is off
	movwf	CCP1CON				; CCP module if off
	movwf	SSPCON				; Serial port disabled
	movwf	RCSTA				; AUSART disabled

;	Bank 1 items next
	banksel	TRISA
	movlw	B'11101111'
	movwf	TRISA				; Port A bit 4 is output. Rest are inputs
	clrf	ANSEL				; Analog I/O config: all digital I/O
	movlw	B'11110111'			
	movwf	TRISB				; Port B bit 3 is outputs, rest inputs (RB6 & 7: TMR1 clock ignores TRIS values and treats them as inputs) 
	clrf	ADCON1				; Setup ADCON1: Vref from Vdd and Vss
	movlw	B'00100000'
	movwf	OSCCON				; Use internal oscillator at 250 KHz, no oscillator switching (boot setting is 31.25 KHz)
	movlw	B'00000000'
	movwf	PIE1				; Disable all peripheral interupts (including A/D interupt) for now
	movwf	PIE2				; And disable EEPROM write interupt
	movwf	CVRCON				; Switch off comparator voltage reference module
	movlw	B'00000111'			; Switch off comparator module
	movwf	CMCON
	movlw	B'10000000'			; Set OPTION_REG
	movwf	OPTION_REG			; Pull ups off for port B, RB0 interupt on falling edge, other bits for TMR0 which is not used

;	Bank 2 items next (nothing needed to initialise in bank 3)
	banksel	WDTCON
	clrf	WDTCON				; Disable watchdog timer

;	Next, initialise port output values (do this after setting I/O config)
	banksel	PORTA				; All ports in same page (0)
	clrf	PORTA				; PORTA outputs (0, 4) low
	bcf		PORTB,SCL			; Set PORTB output 3 low (if SCL is low, no start or stop possible)

;	Initialise key variables (initialise seconds and minutes etc)
	movlw	D'15'
	movwf	secs_4				; initialise second and minute counters
	movlw	LOG_FREQ
	movwf	mins
	clrf	rainfall			; initialise rainfall counter
	clrf	ee					; loop around counter for serial eeprom write number
	clrf	gg					; clear flag for error communicating with DS1621 sensor
	clrf	t_flag				; clear flag which controls when temp is read (see interupt clock routine)
	clrf	clk_adj				; clear flag for clock correction routine
	movlw	HRS24				; set 24 hour counter
	movwf	elap24
	call	read_cnt			; ensure ram values of record counter (low and high) are updated


; Now clear TMR1 and enable interupts incl TMR1 interupt (this will allow TMR1 interupt to occur)
; and initialise the DS1621 temperature sensor. TMR1 used as real time clock

	banksel	PORTA				; select bank 0
	clrf	TMR1L				; Initialise timer 1
	clrf	TMR1H				; With 1:2 prescaler, will generate interupt every 4 sec with 32.768 KHz Xtal
	clrf	PIR1				; clear all peripheral interupts (incl TMR1)
	banksel	PIE1
	bsf		PIE1, TMR1IE		; enable TMR1 interupts
	bsf		INTCON, PEIE
	bsf		INTCON, GIE			; all done - all other code executed from interupt routine

	movlw	D'5'				; initialise DS1621 temperature sensor, but first...
	movwf	bb					; set up counter for initialisation cycles (in case error on first go)
ds1621
	call	init1621			; initialise DS1621 to one-shot temperature mode and check for errors
	addlw	D'1'				; did return occur with W containing 0xff? (ie no error)
	btfsc	STATUS,Z
	goto	zzz					; no error: finish up
	decfsz	bb,1				; error: check write loop repeat counter and repeat write
	goto	ds1621				; if necessary (if fail after five repeats, set error flag and continue)



;***************************************************************
zzz			; routine which sleeps the PIC when all other processing done
	sleep
	nop							; instruction after sleep is executed on wake: need to handle cleanly
	nop
	nop
	nop
	nop
	nop
	goto	zzz


;***************************************************************
wait_20		; routine to waste about 20 msec with 250 KHz clock (~1.2 msec @ 4 MHz)
			; aa register is in all pages so no need to select page
	clrf	aa					; set loop counter
not_done
	nop
	nop
	decfsz	aa,1				; decrement loop counter and test
	goto	not_done
	return


;***************************************************************
ee_write	; routine to write data to EERAM. Caller must set EEADR and EEDATA registers before calling
			; return with ram page 0 set
	banksel	EECON1
	bcf		EECON1,EEPGD		; go through the write sequence
	bsf		EECON1,WREN
	movlw	0x55
	movwf	EECON2
	movlw	0xaa
	movwf	EECON2
	bsf		EECON1,WR
	bcf		EECON1,WREN			; disable further writes once write process started
	banksel	PIR2				; check write completed (not really necessary
	btfss	PIR2,EEIF			; since normal program execution can occur during write operation
	goto	$-1
	bcf		PIR2,EEIF			; write done: clear flag bit
	return


;***************************************************************
read_cnt	; routine to read record high and low bytes from eeram: called once per logging interval by clock
			; before count value is updated.
	banksel	EEADR
	movlw	RECNUM_L			; Address for count (low byte)
	movwf	EEADR
	banksel	EECON1				; Read from data memory, not program memory
	bcf		EECON1, EEPGD
	bsf		EECON1, RD			; Now do the read
	banksel	EEDATA				; Read the result
	movf	EEDATA,0
	movwf	rec_num_l			; and place in RAM copy of the count (low byte)
	incf	EEADR,1				; point to count (high byte)
	banksel	EECON1				; Read from data memory, not program memory
	bcf		EECON1, EEPGD
	bsf		EECON1, RD			; Now do the read
	banksel	EEDATA				; Read the result
	movf	EEDATA,0			; and place in RAM copy of count (high byte)
	movwf	rec_num_h
	return


;***************************************************************
write24c	; Routine to write to the 24Cxxx or 24LCxxx serial eeprom
			; Caller must place data bytes in temp_l, temp_h and rainfall
			; rec_num_l and rec_num_h are used as low and high byte start addresses
			; Routine uses block write to avoid complexity (must avoid crossing
			; page boundary during a block write or wrap around will occur (64 bytes per page)
			; 4 bytes written when routine is finished in following order: rainfall, temp_h, temp_l, count
			; (count is value in register ee is is a simple sequence number that rolls over every
			; 256.  Note that a brownout or reboot will reset this counter so such events
			; can be detected)
	banksel	PORTB
	bcf		PORTB,SCL			; ensure clock line low
	banksel	TRISB
	movlw	B'11110101'			; Now, set SDA and SCL lines as outputs (PORTB:1 and 3)
	movwf	TRISB				; SCL should be output anyway, but do this after making sure it's low to be safe
	banksel	PORTB
	bsf		PORTB,SDA			; ensure PORTB:1 (SDA) is high
	bsf		PORTB,SCL			; Get ready to send start signal to EEPROM: first send clock high
	bcf		PORTB,SDA			; Now send SDA low while clock is high (= start)
	bcf		PORTB,SCL			; send clock low now start initiated
	movlw	B'10100000'			; place initiation/control byte to transmit into cc (last bit is zero = write operation)
	movwf	cc
	btfsc	rec_num_h,7			; check if need to write to second eeprom (ie rec_num_h >127)
	bsf		cc,1				; yes: set correct address bit to write to second eeprom (bit 1 = A0 value of eeprom address)
	call	sendbyte			; regardless... send the byte to the eeprom
	addlw	D'1'				; check if error returned by sendbyte
	btfss	STATUS,Z
	retlw	D'100'				; return with error
	movf	rec_num_h,0			; returned with no error: send high byte of address
	movwf	cc
	call	sendbyte
	addlw	D'1'				; check if error in sendbyte
	btfss	STATUS,Z
	retlw	D'100'				; return with error
	movf	rec_num_l,0			; returned with no error: send low byte of address
	movwf	cc
	call	sendbyte
	addlw	D'1'				; check if error in sendbyte
	btfss	STATUS,Z
	retlw	D'100'				; return with error
	movf	ee,0				; returned with no error: first, write count of records (rolls over every 256)
	movwf	cc					; block of four bytes sent to ensure write never crosses page boundary
	call	sendbyte
	addlw	D'1'				; check if error in sendbyte
	btfss	STATUS,Z
	retlw	D'100'				; return with error
	movf	rainfall,0			; returned with no error: now send rainfall byte
	movwf	cc
	call	sendbyte
	addlw	D'1'				; check if error in sendbyte
	btfss	STATUS,Z
	retlw	D'100'				; return with error
	movf	temp_h,0			; returned with no error: send data to write (temp high byte next)
	movwf	cc
	call	sendbyte
	addlw	D'1'				; check if error in sendbyte
	btfss	STATUS,Z
	retlw	D'100'				; return with error
	movf	temp_l,0			; returned with no error: send data to write (now temp low byte)
	movwf	cc
	call	sendbyte
	addlw	D'1'				; check if error in sendbyte
	btfss	STATUS,Z
	retlw	D'100'				; return with error
	banksel	TRISB				; set SDA to output (sendbyte returns with it as an input)
	bcf		TRISB,SDA
	banksel	PORTB				; Set page back to page 0
	bcf		PORTB,SDA			; First send SDA low (clock should be low at moment)
	bsf		PORTB,SCL			; now send clock high
	bsf		PORTB,SDA			; Now send SDA high = stop condition (initiates write)
	bcf		PORTB,SCL			; now send clock low
								; no need to wait for completion as no write for next hour
; exit routine
	banksel	TRISB				; Set SDA (PORTB:1) as input (pulled high by pullup)
	bsf		TRISB,SDA			; Leave SCL (PORTB:3) as output
	bcf		PIE1,3				; ensure serial port interupt disabled
	retlw	D'255'				; return with no error


;***************************************************************
sendbyte	; Routine to send a byte to the serial eeprom.  Byte to send must be placed into cc by caller
			; This routine also checks for ACK at end of transmission. Caller should start with SCL low.
			; Routine exits with SCL low and SDA as input.
			; Assumed that PIC clock no faster than 4 MHz (1 MHz instruction rate) and
			; serial eeprom capable of running at 400 KHz clock: 2 instructions (2 usec or 500 KHz) for
			; two instructions needed to send clock high then low with PIC oscillator at 4 MHz
	movlw	D'8'
	movwf	dd					; set counter for number of bits to send
	banksel	TRISB				; Set SDA (PORTB:1) as an output (pulled high by pullup)
	bcf		TRISB,SDA
	banksel	PORTB
nxt_bit
	bsf		PORTB,SDA			; Set SDA high
	rlf		cc,1				; Load first bit to send
	btfss	STATUS,C			; check if zero or one to send
	bcf		PORTB,SDA			; Send zero (or jump to send 1 as SDA already high)
	bsf		PORTB,SCL			; send clock high then low to load bit to serial eeprom
	nop							; waste time
	nop
	bcf		PORTB,SCL
	decfsz	dd,1				; Last bit sent?
	goto	nxt_bit				; No, do next bit
; all bits done: check for ACK from serial eeprom
	bsf		PORTB,SDA			; Ensure SDA latch is high (shouldn't need too though as next TRISB should fix)
	banksel	TRISB
	bsf		TRISB,SDA			; Set SDA high and set as input (so eeprom can pull it low if it ACKs)
	nop
	banksel	PORTB
	bsf		PORTB,SCL			; send clock high then check if serial eeprom pulled SDA low
	nop
	nop
	btfsc	PORTB,SDA
	goto	ack_err				; no ACK: error: return with error value
	nop
	bcf		PORTB,SCL			; ACK received: send clock low
	retlw	D'255'				; and return with no error
ack_err
	bcf		PORTB,SCL			; no ACK: send clock low
	retlw	D'100'				; and return with error value


;***************************************************************
read_ack		; Routine to read a byte from an I2C slave (EEPROM or DS1621)
				; This version sends an ACK from the master when byte received
				; Start with SCL low and SDA set as an input (default states)
				; bit 7 is transmitted first
				; The byte just read is placed in register cc
				; Assumed that PIC clock no faster than 4 MHz (1 MHz instruction rate) and
				; serial eeprom capable of running at 400 KHz clock: 2 instructions (2 usec or 500 KHz) for
				; two instructions needed to send clock high then low with PIC oscillator at 4 MHz
	movlw	D'8'
	movwf	dd					; set counter for number of bits to send
	banksel	PORTB				; Ensure on page 0
rd_nxt
	bcf		STATUS,C			; initialise bit for receiving data
	bsf		PORTB,SCL			; send clock high then low to signal slave to send bit
	nop
	nop
	btfsc	PORTB,SDA			; now check bit sent by transmitter
	bsf		STATUS,C			; set bit if 'one' received otherwise continue as bit already set to zero
	rlf		cc,1				; move received bit to register cc:0 and shift previous bits one place to left
	bcf		PORTB,SCL			; send clock low so transmitter can place next bit
	decfsz	dd,1				; Last bit recd?
	goto	rd_nxt				; No, read next bit
; all bits done: master sends ACK
	banksel	TRISB
	bcf		TRISB,SDA			; Set SDA as an output (to enable master to send ACK)
	banksel	PORTB
	bcf		PORTB,SDA			; Ensure SDA is low
	nop
	nop
	bsf		PORTB,SCL			; now send clock high (while SDA low = ACK)
	nop							; waste time
	nop
	bcf		PORTB,SCL			; then send clock low
	nop
	banksel	TRISB				; finally reset SDA as input
	bsf		TRISB,SDA
	banksel	PORTB				; Set page back to page 0
	return						; and return with received byte in register cc



;***************************************************************
rd_noack		; Routine to read a byte from an I2C slave (EEPROM or DS1621)
				; This version sends a 'no' ACK from the master when byte received
				; Start with SCL low and SDA set as an input (default states)
				; bit 7 is transmitted first
				; The byte just read is placed in register cc
				; Assumed that PIC clock no faster than 4 MHz (1 MHz instruction rate) and
				; serial eeprom capable of running at 400 KHz clock: 2 instructions (2 usec or 500 KHz) for
				; two instructions needed to send clock high then low with PIC oscillator at 4 MHz
	movlw	D'8'
	movwf	dd					; set counter for number of bits to send
	banksel	PORTB				; Ensure on page 0
rd_nxtb
	bcf		STATUS,C			; initialise bit for receiving data
	bsf		PORTB,SCL			; send clock high then low to signal slave to send bit
	nop
	nop
	btfsc	PORTB,SDA			; now check bit sent by transmitter
	bsf		STATUS,C			; set bit if 'one' received
	rlf		cc,1				; move received bit to register cc and shift previous bits one place to left
	bcf		PORTB,SCL			; send clock low so transmitter can place next bit
	decfsz	dd,1				; Last bit recd?
	goto	rd_nxtb				; No, read next bit
; all bits done: master sends 'no' ACK
	banksel	TRISB
	bcf		TRISB,SDA			; Set SDA as an output (to enable master to send no ACK)
	banksel	PORTB
	bsf		PORTB,SDA			; Ensure SDA is high
	nop
	nop
	bsf		PORTB,SCL			; now send clock high (while SDA high = 'no' ACK)
	nop							; waste time
	nop
	bcf		PORTB,SCL			; then send clock low
	nop
	banksel	TRISB				; finally reset SDA as input
	bsf		TRISB,SDA
	banksel	PORTB				; Set page back to page 0
	return						; and return


;***************************************************************
init1621	; Routine to initialise the DS1621 temperature sensor
			; Register gg used as an error flag (error = bit 4 set)
	banksel	PORTB
	bcf		PORTB,SCL			; ensure clock line low
	banksel	TRISB
	movlw	B'11110101'			; Now, set SDA and SCL lines as outputs (PORTB:1 and 3)
	movwf	TRISB				; SCL should be low anyway, but do this after making sure it's low to be safe
	banksel	PORTB
	bsf		PORTB,SDA			; ensure PORTB:1 (SDA) is high
	bsf		PORTB,SCL			; Get ready to send start signal to DS1621: first send clock high
	bcf		PORTB,SDA			; Now send SDA low while clock is high (= start)
	bcf		PORTB,SCL			; send clock low now start initiated
	movlw	B'10010000'			; place initiation/control byte to transmit into cc (last bit is zero = write operation)
	movwf	cc
	call	sendbyte			; send the byte to the DS1621
	addlw	D'1'				; check if error returned by sendbyte
	btfsc	STATUS,Z
	goto	$+3
	bsf		gg,4				; set communication error flag
	retlw	D'100'				; return with error
	movlw	0xAC				; send command to write to the control register
	movwf	cc
	call	sendbyte			; send the byte to the DS1621
	addlw	D'1'				; check if error returned by sendbyte
	btfsc	STATUS,Z
	goto	$+3
	bsf		gg,4				; set communication error flag
	retlw	D'100'				; return with error
	movlw	B'10001001'			; send data for control register (one shot temperature mode: bit 0, rest don't matter)
	movwf	cc
	call	sendbyte			; send the byte to the DS1621
	addlw	D'1'				; check if error returned by sendbyte
	btfsc	STATUS,Z
	goto	$+3
	bsf		gg,4				; set communication error flag
	retlw	D'100'				; return with error
	banksel	TRISB				; Set SDA as output (returns from sendbyte as an input)
	bcf		TRISB,SDA
	banksel	PORTB				; if no error, send stop condition to finish
	bcf		PORTB,SDA			; First send SDA low (clock should be low at moment)
	bsf		PORTB,SCL			; now send clock high
	bsf		PORTB,SDA			; Now send SDA high = stop condition
	bcf		PORTB,SCL			; now send clock low
	banksel	TRISB				; Set SDA (PORTB:1) as input (pulled high by pullup)
	bsf		TRISB,SDA			; Leave SCL (PORTB:3) as output
	bcf		PIE1,3				; ensure serial port interupt disabled
	clrf	gg					; clear error flag
	retlw	D'255'				; return with no error


;***************************************************************
t_log		; Routine to initiate a one-off recording of current temperature in the DS1621
	banksel	PORTB
	bcf		PORTB,SCL			; ensure clock line low
	banksel	TRISB
	movlw	B'11110101'			; Now, set SDA and SCL lines as outputs (PORTB:1 and 3)
	movwf	TRISB				; SCL should be low anyway, but do this after making sure it's low to be safe
	banksel	PORTB
	bsf		PORTB,SDA			; ensure PORTB:1 (SDA) is high
	bsf		PORTB,SCL			; Get ready to send start signal to DS1621: first send clock high
	bcf		PORTB,SDA			; Now send SDA low while clock is high (= start)
	bcf		PORTB,SCL			; send clock low now start initiated
	movlw	B'10010000'			; place initiation/control byte to transmit into cc (last bit is zero = write operation)
	movwf	cc
	call	sendbyte			; send the byte to the DS1621
	addlw	D'1'				; check if error returned by sendbyte
	btfsc	STATUS,Z
	goto	$+3
	bsf		gg,4				; set communication error flag
	retlw	D'100'				; return with error
	movlw	0xEE				; send command to record temperature
	movwf	cc
	call	sendbyte			; send the byte to the DS1621
	addlw	D'1'				; check if error returned by sendbyte
	btfsc	STATUS,Z
	goto	$+3
	bsf		gg,4				; set communication error flag
	retlw	D'100'				; return with error
	banksel	TRISB				; Otherwise, finish: Set SDA as output (returns from sendbyte as an input)
	bcf		TRISB,SDA
	banksel	PORTB				; if no error, send stop condition which will start temp reading
	bcf		PORTB,SDA			; First send SDA low (clock should be low at moment)
	bsf		PORTB,SCL			; now send clock high
	bsf		PORTB,SDA			; Now send SDA high = stop condition
	bcf		PORTB,SCL			; now send clock low
								; no need to wait for completion - clock will read results in 4 seconds
	banksel	TRISB				; Set SDA (PORTB:1) as input (pulled high by pullup)
	bsf		TRISB,SDA			; Leave SCL (PORTB:3) as output
	bcf		PIE1,3				; ensure serial port interupt disabled
	clrf	gg					; clear error flag
	retlw	D'255'				; return with no error


;***************************************************************
t_read		; Routine to recall last read temperature from DS1621
	banksel	PORTB
	bcf		PORTB,SCL			; ensure clock line low
	banksel	TRISB
	movlw	B'11110101'			; Now, set SDA and SCL lines as outputs (PORTB:1 and 3)
	movwf	TRISB				; SCL should be low anyway, but do this after making sure it's low to be safe
	banksel	PORTB
	bsf		PORTB,SDA			; ensure PORTB:1 (SDA) is high
	bsf		PORTB,SCL			; Get ready to send start signal to DS1621: first send clock high
	bcf		PORTB,SDA			; Now send SDA low while clock is high (= start)
	bcf		PORTB,SCL			; send clock low now start initiated
	movlw	B'10010000'			; place initiation/control byte to transmit into cc (last bit is zero = write operation)
	movwf	cc					; write is done first to 'write' the control command to the DS1621
	call	sendbyte			; send the byte to the DS1621
	addlw	D'1'				; check if error returned by sendbyte
	btfsc	STATUS,Z
	goto	$+3
	bsf		gg,4				; set communication error flag
	retlw	D'100'				; return with error
	movlw	0xAA				; send command to read temperature
	movwf	cc
	call	sendbyte			; send the byte to the DS1621
	addlw	D'1'				; check if error returned by sendbyte
	btfsc	STATUS,Z
	goto	$+3
	bsf		gg,4				; set communication error flag
	retlw	D'100'				; return with error
;	now ready to read the two bytes transmitted and ACK them
	banksel	TRISB				; Set SDA as output (returns from sendbyte as an input)
	bcf		TRISB,SDA
	banksel	PORTB
	bsf		PORTB,SDA			; need to send new START: first ensure PORTB:1 (SDA) is high
	bsf		PORTB,SCL			; then send clock high
	bcf		PORTB,SDA			; Now send SDA low while clock is high (= start)
	bcf		PORTB,SCL			; send clock low now start initiated
	movlw	B'10010001'			; place initiation/control byte to transmit into cc (last bit is one = read operation)
	movwf	cc
	call	sendbyte			; send the byte to the DS1621
	addlw	D'1'				; check if error returned by sendbyte
	btfsc	STATUS,Z
	goto	$+3
	bsf		gg,4				; set communication error flag
	retlw	D'100'				; return with error
	call	read_ack			; read the first byte (most significant byte)
	movf	cc,0				; copy result to temperature high byte (whole number portion of temperature)
	movwf	temp_h
	call	rd_noack			; read second byte (temperature low byte)
	movf	cc,0
	movwf	temp_l
	banksel	TRISB				; Set SDA as output (returns from sendbyte as an input)
	bcf		TRISB,SDA			; in preparation for STOP
	banksel	PORTB				; if no error, send stop condition to finish
	bcf		PORTB,SDA			; First send SDA low (clock should be low at moment)
	bsf		PORTB,SCL			; now send clock high
	bsf		PORTB,SDA			; Now send SDA high = stop condition
	bcf		PORTB,SCL			; now send clock low

;	now read counter and slope
	bsf		PORTB,SCL			; Get ready to send start signal to DS1621: first send clock high
	bcf		PORTB,SDA			; Now send SDA low while clock is high (= start)
	bcf		PORTB,SCL			; send clock low now start initiated
	movlw	B'10010000'			; place initiation/control byte to transmit into cc (last bit is zero = write operation)
	movwf	cc					; write is done first to 'write' the control command to the DS1621
	call	sendbyte			; send the byte to the DS1621
	addlw	D'1'				; check if error returned by sendbyte
	btfsc	STATUS,Z
	goto	$+3
	bsf		gg,4				; set communication error flag
	retlw	D'100'				; return with error
	movlw	0xA8				; send command to read counter
	movwf	cc
	call	sendbyte			; send the byte to the DS1621
	addlw	D'1'				; check if error returned by sendbyte
	btfsc	STATUS,Z
	goto	$+3
	bsf		gg,4				; set communication error flag
	retlw	D'100'				; return with error
;	now ready to read the byte transmitted and NO ACK
	banksel	TRISB				; Set SDA as output (returns from sendbyte as an input)
	bcf		TRISB,SDA
	banksel	PORTB
	bsf		PORTB,SDA			; need to send new START: first ensure PORTB:1 (SDA) is high
	bsf		PORTB,SCL			; then send clock high
	bcf		PORTB,SDA			; Now send SDA low while clock is high (= start)
	bcf		PORTB,SCL			; send clock low now start initiated
	movlw	B'10010001'			; place initiation/control byte to transmit into cc (last bit is one = read operation)
	movwf	cc
	call	sendbyte			; send the byte to the DS1621
	addlw	D'1'				; check if error returned by sendbyte
	btfsc	STATUS,Z
	goto	$+3
	bsf		gg,4				; set communication error flag
	retlw	D'100'				; return with error
	call	rd_noack			; read counter byte
	movf	cc,0
	movwf	ff					; use ff to store counter value	
	banksel	TRISB				; Set SDA as output (returns from sendbyte as an input)
	bcf		TRISB,SDA			; in preparation for STOP
	banksel	PORTB				; if no error, send stop condition to finish
	bcf		PORTB,SDA			; First send SDA low (clock should be low at moment)
	bsf		PORTB,SCL			; now send clock high
	bsf		PORTB,SDA			; Now send SDA high = stop condition
	bcf		PORTB,SCL			; now send clock low

	bsf		PORTB,SCL			; Get ready to send start signal to DS1621: first send clock high
	bcf		PORTB,SDA			; Now send SDA low while clock is high (= start)
	bcf		PORTB,SCL			; send clock low now start initiated
	movlw	B'10010000'			; place initiation/control byte to transmit into cc (last bit is zero = write operation)
	movwf	cc					; write is done first to 'write' the control command to the DS1621
	call	sendbyte			; send the byte to the DS1621
	addlw	D'1'				; check if error returned by sendbyte
	btfsc	STATUS,Z
	goto	$+3
	bsf		gg,4				; set communication error flag
	retlw	D'100'				; return with error
	movlw	0xA9				; send command to read slope
	movwf	cc
	call	sendbyte			; send the byte to the DS1621
	addlw	D'1'				; check if error returned by sendbyte
	btfsc	STATUS,Z
	goto	$+3
	bsf		gg,4				; set communication error flag
	retlw	D'100'				; return with error
;	now ready to read the byte transmitted and NO ACK
	banksel	TRISB				; Set SDA as output (returns from sendbyte as an input)
	bcf		TRISB,SDA
	banksel	PORTB
	bsf		PORTB,SDA			; need to send new START: first ensure PORTB:1 (SDA) is high
	bsf		PORTB,SCL			; then send clock high
	bcf		PORTB,SDA			; Now send SDA low while clock is high (= start)
	bcf		PORTB,SCL			; send clock low now start initiated
	movlw	B'10010001'			; place initiation/control byte to transmit into cc (last bit is one = read operation)
	movwf	cc
	call	sendbyte			; send the byte to the DS1621
	addlw	D'1'				; check if error returned by sendbyte
	btfsc	STATUS,Z
	goto	$+3
	bsf		gg,4				; set communication error flag
	retlw	D'100'				; return with error
	call	rd_noack			; read slope byte
	movf	cc,0
	movwf	hh					; use hh to store slope value
	banksel	TRISB				; Set SDA as output (returns from sendbyte as an input)
	bcf		TRISB,SDA			; in preparation for STOP
	banksel	PORTB				; if no error, send stop condition to finish
	bcf		PORTB,SDA			; First send SDA low (clock should be low at moment)
	bsf		PORTB,SCL			; now send clock high
	bsf		PORTB,SDA			; Now send SDA high = stop condition
	bcf		PORTB,SCL			; now send clock low
	banksel	TRISB				; Set SDA (PORTB:1) as input (pulled high by pullup)
	bsf		TRISB,SDA			; Leave SCL (PORTB:3) as output
	bcf		PIE1,3				; ensure serial port interupt disabled
;	finally convert the counter and slope values to the fractional portion of temperature
;	and adjust the high byte if necessary:
;	temperature = temp_h - 0.25 + (slope - counter)/slope
	bcf		STATUS,C			; ensure carry bit clear first (for rlf instructions)
	rlf		ff,1				; multiply both counter and slope by 4 (if possible) to improve accuracy
	rlf		ff,0				; put second result in W in case first needs to be recovered
	bcf		STATUS,C			; ensure carry bit clear again (for rlf instructions)
	rlf		hh,1				; (typical value for slope is 0x2E at 20 C)
	rlf		hh,1
	btfss	STATUS,C			; did second multiplication by 2 result in slope overflow?
	goto	t_calc				; no: all OK, jump and continue conversion
	rrf		hh,1				; Yes, overflow: reverse second (rlf) multiplication for slope and counter
	movf	ff,0				; Place result of first division by two for count into W to correct (see next instruction too)
t_calc
	movwf	ff					; And store correct value in ff (also still in W), then subtract count from slope (count always smaller)
	subwf	hh,0				; to get correct count value
	movwf	ff					; store result back in count register
	clrf	cc					; clear register for result of division
	movlw	D'10'				; divide slope by 10 to find counts per 0.1 C (approx)
div_10
	subwf	hh,1
	incf	cc,1				; keep track of division result (won't affect carry)
	btfsc	STATUS,C			; Carry flag only set if value in hh was > 10 before subtraction
	goto	div_10				; Repeat until result goes negative
	decf	cc,1				; fix value in cc since one too many subtractions of 10 done
	clrf	temp_l				; prepare for final calculation
	movf	cc,0				; divide the corrected count by the count per C (latter in cc)
div_c
	subwf	ff,1
	incf	temp_l,1			; and update temp_l value
	btfsc	STATUS,C			; Carry flag will remain set until subtraction produces negative result in ff
	goto	div_c
	movlw	D'3'				; exits division loop after one too many subtractions: correct in next step by subtract 3 instead of 2
	subwf	temp_l,1			; subtract 0.2 C to approximate - 0.25 C correction (use 3 to include correction from previous step)
	btfsc	STATUS,C			; was result positive or negative (carry clear if negative)
	goto	$+4					; not negative: finished: return with no error
	decf	temp_h,1			; if negative, decrement whole number portion of temperature to correct
	movlw	D'10'
	addwf	temp_l,1			; and correct the value in temp_l
;	done: return
	clrf	gg					; clear error flag
	retlw	D'255'				; return with no error


;***************************************************************
dump		; Routine retrieve contents of eeproms, process into csv data and send to serial port
			; This routine should be entered before data logging on computer is activated
			; because the process of turning on the AUSART module in the PIC sends one spurious character
			; before the transmit line settles to a stable high state.
	banksel	SPBRG
	movlw	D'12'
	movwf	SPBRG				; set up baud rate to 19,200 bps (TXSTA in high speed mode)
	movlw	B'00000100'			; set up AUSART transmit control (in same page as OSCCON)
	movwf	TXSTA				; 8 bit mode (bit 6), transmit on (bit 5), asynchronous mode (bit 4), high speed mode (bit 2)
	bcf		PIE1,TXIE			; ensure serial port interupts disabled (not really necesary because this
	bcf		PIE1,RCIE			; routine called from interupt routine which disables GIE)
	banksel	RCSTA
	movlw	B'10000000'			; enable serial port and set up AUSART receive control (none planned anyway)
	movwf	RCSTA				; AUSART on (bit 7), 8 bit reception (bit 6), continuous and single reception off (bits 5 & 4)
	banksel	TXSTA
	bsf		TXSTA,5				; enable transmission
	banksel	TXREG
	call	wait_20
	bsf		PORTA,4				; turn LED on continuously when ready to transmit
	call	wait_20
	btfss	PORTA,0				; Ensure MAX232 interface still powered up
	goto	quit_dmp			; If not, clean up and quit
	btfsc	PORTA,1				; Check if computer ready to receive data
	goto	$-4					; loop until ready
	call	rd_init				; sets both eeprom address counters to 0x0000
	addlw	D'1'
	btfss	STATUS,Z			; check if error from initiating eeprom address reset
	goto	rd_fin				; yes, error: abort (user needs to try again)
	call	wait_20				; delays inserted in attempt to get reliable reading on first attempt
	call	wait_20				; after PC first started
	call	wait_20
	call	wait_20
	banksel	OSCCON
	movlw	B'01100000'			; change system oscillator to 4 MHz
	movwf	OSCCON
	call	wait_20				; allow clock to stabilise (not necessary?)
	banksel	PORTA
	movf	rec_num_h,0			; set end address for reading
	movwf	addr_hi				; high byte
	movf	rec_num_l,0			; note the addr_hi and addr_low registers are in page zero (same as PORTA)
	movwf	addr_low			; low byte
	rrf		addr_low,1			; divide counters by 4 since loop below reads four bytes at a time to process
	rrf		addr_low,1			; lower two bits will always be zero
	bcf		addr_low,6			; but clear upper two bits anyway (eg in case carry flag was set before starting)
	bcf		addr_low,7
	rrf		addr_hi,1
	btfsc	STATUS,C			; was bit 0 a "one"?
	bsf		addr_low,6			; yes: set bit 6 of addr_low
	rrf		addr_hi,1
	btfsc	STATUS,C			; was bit 0 a "one"?
	bsf		addr_low,7			; yes: set bit 7 of addr_low
	bcf		addr_hi,7			; finally, fix any carry that occurred into bits 7 and 6 of addr_hi from
	bcf		addr_hi,6			; the last two rrf instructions
	movf	rec_num_l,1			; finally, test if rec_num_l:rec_num_h was 00:00.  If so, set addr_hi:addr_low to 40:00
	btfss	STATUS,Z			; so that loops below will repeat (65,536/4) times (ie force entire contents of both eeproms to be read)
	goto	$+7
	movf	rec_num_h,1
	btfss	STATUS,Z
	goto	$+4
	clrf	addr_low
	movlw	B'01000000'
	movwf	addr_hi
	clrf	loop_i				; prepare loop counters to keep track of eeprom reading
	clrf	loop_o
	clrf	set_no				; also reset counter used to monitor 'breaks' in data logging
	call	tx_cnt				; transmit number of records logged
	call	tx_head				; transmit the header: "rain,temp"
; inner and outer loop here repeats until all data recorded since last reset is sent
; or if rec_num_l and rec_num_h are were reset to zero, read and send entire contents of both eeproms.
; Inner portion of loop (starting at read_4) normally repeats 256 times except on final pass (final pass = when outer loop has repeated
; addr_hi times), at which point the innner loop repeats addr_low times to finish off.
out_loop
	movf	addr_hi,0			; outer loop portion: check if outer loop finished repeating
	subwf	loop_o,0
	btfss	STATUS,Z			; is outer loop counter = addr_hi?
	goto	read_4				; no, read another 4 bytes (with loop_i starting at zero to give 256 passes of inner loop)
	movf	addr_low,0			; yes: first test if there are any bytes left to read (addr_low not zero)
	btfsc	STATUS,Z			; If addr_low not zero, loop_i needs to be set so it repeats addr_low times for final pass
	goto	rd_fin				; addr_low is zero: finished
	movwf	loop_i				; copy addr_low to loop_i
	clrf	addr_low			; and set addr_low and addr_hi so this loop exits at next pass
	incf	addr_hi,1			; loop_o will increment after next inner loop making this correction necessary to force exit after next pass
read_4
	banksel	PORTB				; first, initiate the read operation
	bcf		PORTB,SCL			; ensure clock line low
	banksel	TRISB
	movlw	B'11110101'			; Now, set SDA and SCL lines as outputs (PORTB:1 and 3)
	movwf	TRISB				; SCL should be output anyway, but do this after making sure it's low to be safe
	banksel	PORTB
	bsf		PORTB,SDA			; ensure PORTB:1 (SDA) is high
	nop
	nop
	bsf		PORTB,SCL			; Get ready to send start signal to EEPROM: first send clock high
	nop
	nop
	bcf		PORTB,SDA			; Now send SDA low while clock is high (= start)
	nop
	nop
	bcf		PORTB,SCL			; send clock low now start initiated
	movlw	B'10100001'			; place initiation/control byte to transmit into cc (last bit is one = read operation)
	movwf	cc
	btfsc	loop_o,5			; check if need to read second eeprom
	bsf		cc,1				; yes: set correct address bit to write to second eeprom (bit 1 = A0 value of eeprom address)
	call	sendbyte			; regardless... send the byte to the eeprom
	addlw	D'1'				; check if error returned by sendbyte
	btfss	STATUS,Z
; could try jump to a "flashing routine" here until MAX232 powered down then up again
	goto	rd_fin				; exit (clean up first) if problem communicating
;**************************************
	call	read_ack			; no error: read 4 bytes (returns in page 0)
	movf	cc,0				; move read byte into byte 1
	movwf	rd01
	call	read_ack			; no error: read 4 bytes (returns in page 0)
	movf	cc,0				; move read byte into byte 2
	movwf	rd02
	call	read_ack			; no error: read 4 bytes (returns in page 0)
	movf	cc,0				; move read byte into byte 3
	movwf	rd03
	call	rd_noack			; final byte: no ACK, then issue a stop (returns in page 0)
	movf	cc,0				; move read byte into byte 4
	movwf	rd04
	banksel	TRISB				; set SDA to output (rd_noack returns with it as an input)
	bcf		TRISB,SDA
	banksel	PORTB				; Set page back to page 0
	bcf		PORTB,SDA			; First send SDA low (clock should be low at moment)
	nop
	nop
	bsf		PORTB,SCL			; now send clock high
	nop
	nop
	bsf		PORTB,SDA			; Now send SDA high = stop condition
	nop
	nop
	bcf		PORTB,SCL			; now send clock low
; convert bytes and transmit results
; first, check if all four bytes read are 0xff: transmit "nul" if so
	incfsz	rd01,0				; put all test results in W so as not to corrupt rd0* contents
	goto	chk_rst
	incfsz	rd02,0
	goto	chk_rst
	incfsz	rd03,0
	goto	chk_rst
	incfsz	rd04,0
	goto	chk_rst
	call	tx_nul
	goto	tx_loop
; next, check for a reset during logging and signal user if so by transmitting "BREAK" then the three bytes
chk_rst
	movf	rd01,0				; first of four bytes recorded by weather station is a sequence number
								; and if the next value is not a simple increment, that indicates a
								; fault during recording that caused a reset (or similar) in the weather station
	subwf	set_no,0			; check if eeprom count matches its parallel in this routine
	btfsc	STATUS,Z			; (they should if no break occurred)
	goto	no_brk
	movf	rd01,0				; break occurred: reset software counter to match value in eeprom
	movwf	set_no
	call	tx_brk				; transmit "BREAK,", then process rest of bytes for this location and transmit anyway
								; (user can check after the "break" message if they are valid)
no_brk
; otherwise, transmit the bytes
	movf	rd02,0				; convert and transmit rainfall count
	movwf	cc
	call	b_to_a
	movlw	COMMA				; transmit comma ","
	movwf	TXREG				; move byte to transmit register to start transmission
	nop
	nop
	btfss	PIR1,TXIF			; poll for transmit buffer to empty (can't load next byte until then)
	goto	$-3					; TXIF goes high once byte transfers from TXREG to transmit shift register
	movf	rd03,0				; then temperature high byte
	movwf	cc
	call	b_to_a
	movlw	DOT					; transmit the decimal point "."
	movwf	TXREG				; move byte to transmit register to start transmission
	nop
	nop
	btfss	PIR1,TXIF			; poll for transmit buffer to empty (can't load next byte until then)
	goto	$-3					; TXIF goes high once byte transfers from TXREG to transmit shift register
	movf	rd04,0				; then temperature decimal portion
	movwf	cc
	call	b_to_a
	movlw	CR					; send <carriage return>
	movwf	TXREG				; move byte to transmit register to start transmission
	nop
	nop
	btfss	PIR1,TXIF			; poll for transmit buffer to empty (can't load next byte until then)
	goto	$-3					; TXIF goes high once byte transfers from TXREG to transmit shift register
	movlw	LF					;<line feed>
	movwf	TXREG				; move byte to transmit register to start transmission
	nop
	nop
	btfss	PIR1,TXIF			; poll for transmit buffer to empty (can't load next byte until then)
	goto	$-3					; TXIF goes high once byte transfers from TXREG to transmit shift register
	incf	set_no,1			; done: finally, increment counter that checks for breaks in logging (ready for next 4 bytes read)
; inner and outer loop controls for number of bytes to read and process
tx_loop
	decfsz	loop_i,1			; see if loop_i has decremented to zero (256 passes for all except final pass)
	goto	read_4				; no: keep reading
	incf	loop_o,1			; yes, increment outer loop
	goto	out_loop			; and jump to check if it equals addr_hi yet
;	movf	rd01,0				; transmit the bytes
;	movwf	TXREG				; move byte to transmit register to start transmission
;	nop
;	nop
;	btfss	PIR1,TXIF			; poll for transmit buffer to empty (can't load next byte until then)
;	goto	$-3					; TXIF goes high once byte transfers from TXREG to transmit shift register
;	movf	rd02,0				; transmit the bytes
;	movwf	TXREG				; move byte to transmit register to start transmission
;	nop
;	nop
;	btfss	PIR1,TXIF			; poll for transmit buffer to empty (can't load next byte until then)
;	goto	$-3					; TXIF goes high once byte transfers from TXREG to transmit shift register
;	movf	rd03,0				; transmit the bytes
;	movwf	TXREG				; move byte to transmit register to start transmission
;	nop
;	nop
;	btfss	PIR1,TXIF			; poll for transmit buffer to empty (can't load next byte until then)
;	goto	$-3					; TXIF goes high once byte transfers from TXREG to transmit shift register
;	movf	rd04,0				; transmit the bytes
;	movwf	TXREG				; move byte to transmit register to start transmission
;	nop
;	nop
;	btfss	PIR1,TXIF			; poll for transmit buffer to empty (can't load next byte until then)
;	goto	$-3					; TXIF goes high once byte transfers from TXREG to transmit shift register

; finish up
rd_fin
	banksel	PORTA
	call	wait_20				; allow time for last byte(s) to transmit
	btfss	PORTA,1				; Don't exit until computer logging turned off by user
	goto	$-2					; loop until so
	call	wait_20				; Now wait until user powers down MAX232 interface
	btfsc	PORTA,0
	goto	$-2
	call	wait_20				; debounce the switch opening
	btfsc	PORTA,0
	goto	$-5
quit_dmp
	banksel	RCSTA				; Done: first turn AUSART off
	clrf	RCSTA
	bcf		PORTA,4				; turn LED off when done
	banksel	OSCCON
	movlw	B'00100000'			; change system oscillator back to 250 KHz
	movwf	OSCCON
	clrf	TXSTA				; turn transmit mode off
	return						; and finish




;***************************************************************
rd_init		; Routine sets up 24C eeprom (max clock 400 KHz) for reading
			; Assumed that PIC clock running no faster than 4 MHz by calling routine
			; This routine sets the address pointer in both eeproms to zero
	banksel	PORTB
	bcf		PORTB,SCL			; ensure clock line low
	banksel	TRISB
	movlw	B'11110101'			; Now, set SDA and SCL lines as outputs (PORTB:1 and 3)
	movwf	TRISB				; SCL should be output anyway, but do this after making sure it's low to be safe
	movlw	D'2'
	movwf	ff					; set loop counter (two passes)
rst_addr
	banksel	PORTB
	bsf		PORTB,SDA			; ensure PORTB:1 (SDA) is high
	nop
	nop
	bsf		PORTB,SCL			; Get ready to send start signal to EEPROM: first send clock high
	nop							; waste time
	nop
	bcf		PORTB,SDA			; Now send SDA low while clock is high (= start)
	nop							; waste time
	nop
	bcf		PORTB,SCL			; send clock low now start initiated
	movlw	B'10100000'			; place initiation/control byte to transmit into cc (last bit is zero = write operation)
	movwf	cc
	btfsc	ff,0				; check if need to write to second eeprom (use bit zero of loop counter)
	bsf		cc,1				; yes: set correct address bit to read from second eeprom (bit 1 = A0 value of eeprom address)
	call	sendbyte			; send the byte to the eeprom
	addlw	D'1'				; check if error returned by sendbyte
	btfss	STATUS,Z
	retlw	D'100'				; return with error
	movlw	0x7f				; returned with no error: send high byte of address
	movwf	cc
	call	sendbyte
	addlw	D'1'				; check if error in sendbyte
	btfss	STATUS,Z
	retlw	D'100'				; return with error
	movlw	0xff				; returned with no error: send low byte of address
	movwf	cc
	call	sendbyte
	addlw	D'1'				; check if error in sendbyte
	btfss	STATUS,Z
	retlw	D'100'				; return with error
	banksel	TRISB				; after address sent and ACKed, send a new start signal - this time to read
	movlw	B'11110101'			; First, set SDA and SCL lines as outputs (PORTB:1 and 3)
	movwf	TRISB				; SCL should be output anyway, but SDA won't be
	banksel	PORTB
	bsf		PORTB,SDA			; ensure PORTB:1 (SDA) is high
	nop
	nop
	bsf		PORTB,SCL			; Get ready to send start signal to EEPROM: first send clock high
	nop							; waste time
	nop
	bcf		PORTB,SDA			; Now send SDA low while clock is high (= start)
	nop							; waste time
	nop
	bcf		PORTB,SCL			; send clock low now start initiated
	movlw	B'10100001'			; place initiation/control byte to transmit into cc (last bit is one = read operation)
	movwf	cc
	btfsc	ff,0				; check if need to write to second eeprom (use bit zero of loop counter)
	bsf		cc,1				; yes: set correct address bit to write to second eeprom (bit 1 = A0 value of eeprom address)
	call	sendbyte			; regardless... send the byte to the eeprom
	addlw	D'1'				; check if error returned by sendbyte
	btfss	STATUS,Z
	retlw	D'100'				; return with error
	call	rd_noack			; read with no ACK (ignore results).  EEPROM address pointer will now be 0x0000
; finish up
	banksel	TRISB				; set SDA to output (rd_noack returns with it as an input)
	bcf		TRISB,SDA
	banksel	PORTB				; Set page back to page 0
	bcf		PORTB,SDA			; First send SDA low (clock should be low at moment)
	nop
	nop
	bsf		PORTB,SCL			; now send clock high
	nop
	nop
	bsf		PORTB,SDA			; Now send SDA high = stop condition
	nop
	nop
	bcf		PORTB,SCL			; now send clock low
	decfsz	ff,1				; do this loop total of two times (one for each eeprom)
	goto	rst_addr
; exit routine
	banksel	TRISB				; Set SDA (PORTB:1) as input (pulled high by pullup)
	bsf		TRISB,SDA			; Leave SCL (PORTB:3) as output
	bcf		PIE1,3				; ensure serial port interupt disabled
	banksel	PORTB				; return in page 0
	retlw	D'255'				; return with no error



;***************************************************************
b_to_a		; Routine to convert binary byte to ascii characters then transmit them.  Up to three characters
			; generated.  Register cc contains byte to convert.
			; Characters generated are placed in registers dig1, dig2 and dig3 (in page 0)
			; and transmitted when done
	banksel	PORTA				; ensure in page 0
	clrf	dig1				; prepare receiving registers
	clrf	dig2
	clrf	dig3
	movf	cc,1				; first check if byte is zero
	btfss	STATUS,Z
	goto	not_z				; no: process and transmit
	movlw	0x30				; yes: transmit "0"
	movwf	TXREG				; move byte to transmit register to start transmission
	nop
	nop
	btfss	PIR1,TXIF			; poll for transmit buffer to empty (can't load next byte until then)
	goto	$-3					; TXIF goes high once byte transfers from TXREG to transmit shift register
	return						; when transmitted, exit
not_z
	movlw	D'10'
	subwf	cc,1				; repeatedly subtract 10 until negative to get raw tens digit
	btfss	STATUS,C			; Carry flag set if result is still = or > 0
	goto	$+3					; Result negative: store ones digit and finish processing tens digit
	incf	dig2,1				; keep track of tens digit
	goto	$-4					; keep looping until result negative (or zero)
	addwf	cc,1				; result of subtract was negative: correct result by adding back 10
	movf	cc,0				; store ones digit after converting to ascii
	addlw	0x30
	movwf	dig3				; ones digit done
	movf	dig2,1				; check if raw tens digit was zero
	btfsc	STATUS,Z
	goto	tx_dig3				; Yes: done, transmit only dig3
	movlw	D'10'				; No: continue to convert tens and 100s digits
	subwf	dig2,1
	btfss	STATUS,C
	goto	$+3					; Result negative: store tens digit and finish processing hundreds digit
	incf	dig1,1				; keep track of hundreds digit
	goto	$-4					; keep looping until result negative (or zero)
	addwf	dig2,1				; result of subtract was negative: correct result by adding 10 back
	movf	dig2,0				; store tens digit after converting to ascii
	addlw	0x30
	movwf	dig2
	movf	dig1,1
	btfsc	STATUS,Z			; Is hundreds digit zero?
	goto	tx_dig2				; Yes: jump to transmit 10s and ones digits only
	movf	dig1,0				; No: convert hundreds digit then transmit them all
	addlw	0x30
	movwf	dig1
	movwf	TXREG				; move dig1 to transmit register to start transmission
	nop
	nop
	btfss	PIR1,TXIF			; poll for transmit buffer to empty (can't load next byte until then)
	goto	$-3					; TXIF goes high once byte transfers from TXREG to transmit shift register
tx_dig2
	movf	dig2,0				; transmit dig2
	movwf	TXREG				; move byte to transmit register to start transmission
	nop
	nop
	btfss	PIR1,TXIF			; poll for transmit buffer to empty (can't load next byte until then)
	goto	$-3					; TXIF goes high once byte transfers from TXREG to transmit shift register
tx_dig3
	movf	dig3,0				; transmit dig3
	movwf	TXREG				; move byte to transmit register to start transmission
	nop
	nop
	btfss	PIR1,TXIF			; poll for transmit buffer to empty (can't load next byte until then)
	goto	$-3					; TXIF goes high once byte transfers from TXREG to transmit shift register
	return						; all done



;***************************************************************
tx_cnt		; Routine to transmit the count of records being transmitted as "records",addr_hi:addr_low (converted to decimal)
			; <carriage return><line feed>"
	banksel	PORTA				; ensure in page 0
	movlw	0x52				;"R"
	movwf	TXREG				; move byte to transmit register to start transmission
	nop
	nop
	btfss	PIR1,TXIF			; poll for transmit buffer to empty (can't load next byte until then)
	goto	$-3					; TXIF goes high once byte transfers from TXREG to transmit shift register
	movlw	0x65				;"e"
	movwf	TXREG				; move byte to transmit register to start transmission
	nop
	nop
	btfss	PIR1,TXIF			; poll for transmit buffer to empty (can't load next byte until then)
	goto	$-3					; TXIF goes high once byte transfers from TXREG to transmit shift register
	movlw	0x63				;"c"
	movwf	TXREG				; move byte to transmit register to start transmission
	nop
	nop
	btfss	PIR1,TXIF			; poll for transmit buffer to empty (can't load next byte until then)
	goto	$-3					; TXIF goes high once byte transfers from TXREG to transmit shift register
	movlw	0x6f				;"o"
	movwf	TXREG				; move byte to transmit register to start transmission
	nop
	nop
	btfss	PIR1,TXIF			; poll for transmit buffer to empty (can't load next byte until then)
	goto	$-3					; TXIF goes high once byte transfers from TXREG to transmit shift register
	movlw	0x72				;"r"
	movwf	TXREG				; move byte to transmit register to start transmission
	nop
	nop
	btfss	PIR1,TXIF			; poll for transmit buffer to empty (can't load next byte until then)
	goto	$-3					; TXIF goes high once byte transfers from TXREG to transmit shift register
	movlw	0x64				;"d"
	movwf	TXREG				; move byte to transmit register to start transmission
	nop
	nop
	btfss	PIR1,TXIF			; poll for transmit buffer to empty (can't load next byte until then)
	goto	$-3					; TXIF goes high once byte transfers from TXREG to transmit shift register
	movlw	0x73				;"s"
	movwf	TXREG				; move byte to transmit register to start transmission
	nop
	nop
	btfss	PIR1,TXIF			; poll for transmit buffer to empty (can't load next byte until then)
	goto	$-3					; TXIF goes high once byte transfers from TXREG to transmit shift register
	movlw	COMMA				;","
	movwf	TXREG				; move byte to transmit register to start transmission
	nop
	nop
	btfss	PIR1,TXIF			; poll for transmit buffer to empty (can't load next byte until then)
	goto	$-3					; TXIF goes high once byte transfers from TXREG to transmit shift register
; convert addr_hi:addr_low to ascii
	clrf	dig3				; prepare counter for hundreds value
	movf	addr_hi,0
	btfss	STATUS,Z			; If high byte zero, simply call b_to_a to convert and transmit low byte
	goto	$+5					; no: keep processing
	movf	addr_low,0			; yes: call b_to_a (if addr_hi is zero, addr_low must have a non zero value)
	movwf	cc
	call	b_to_a
	goto	rec_fin	
	movwf	dig4				; continue here if addr_hi is not zero: copy addr_hi (to dig4) and addr_low (to dig5)
	incf	dig4,1				; increment copy of addr_hi so loop below exits correctly
	movf	addr_low,0
	movwf	dig5
	movlw	D'100'				; keep subtracting 100 from low byte til high byte hits zero
	subwf	dig5,1
	btfsc	STATUS,C			; did result go negative (carry not set)?
	goto	$+4					; no: jump and increment hundreds, then keep subtracting
	decfsz	dig4,1				; yes: decrement high byte then check if zero
	goto	$+2					; high byte non-zero: increment hundreds count and keep subtracting 100
	goto	$+3					; high byte zero: finish processing low and high bytes
	incf	dig3,1				; increment hundreds count
	goto	$-7					; and keep looping
	addwf	dig5,1				; exit hundreds calculation: first correct dig5 for one too many subtractions
; At this point, dig3 will contain hundreds count and dig5 the residue less than 100
	movf	dig3,0				; convert and transmit hundreds
	movwf	cc
	call	b_to_a
	clrf	dig4				; clear tens counter
	movlw	D'10'
	subwf	dig5,1				; repeatedly subtract 10 from residue to get tens and ones digits
	btfsc	STATUS,C			; did result go negative?
	goto	$+3					; no: jump to increment tens counter and subtract again
	addwf	dig5,1				; yes: correct ones digit for one too many subtractions
	goto	$+3					; and jump to transmit
	incf	dig4,1				; increment tens
	goto	$-6					; and keep subtracting
	movf	dig4,0				; exit tens and ones routine here: first transmit tens digit
	addlw	0x30				; after converting to ascii
	movwf	TXREG				; move byte to transmit register to start transmission
	nop
	nop
	btfss	PIR1,TXIF			; poll for transmit buffer to empty (can't load next byte until then)
	goto	$-3					; TXIF goes high once byte transfers from TXREG to transmit shift register
	movf	dig5,0				; transmit ones digit
	addlw	0x30				; after converting to ascii
	movwf	TXREG				; move byte to transmit register to start transmission
	nop
	nop
	btfss	PIR1,TXIF			; poll for transmit buffer to empty (can't load next byte until then)
	goto	$-3					; TXIF goes high once byte transfers from TXREG to transmit shift register
; finally finish up
rec_fin
	movlw	CR					; transmit <carriage return>
	movwf	TXREG				; move byte to transmit register to start transmission
	nop
	nop
	btfss	PIR1,TXIF			; poll for transmit buffer to empty (can't load next byte until then)
	goto	$-3					; TXIF goes high once byte transfers from TXREG to transmit shift register
	movlw	LF					;<line feed>
	movwf	TXREG				; move byte to transmit register to start transmission
	nop
	nop
	btfss	PIR1,TXIF			; poll for transmit buffer to empty (can't load next byte until then)
	goto	$-3					; TXIF goes high once byte transfers from TXREG to transmit shift register
	return



;***************************************************************
tx_head		; Routine to transmit the text "rain,temp<carriage return><line feed>"
	banksel	PORTA				; ensure in page 0
	movlw	0x72				;"r"
	movwf	TXREG				; move byte to transmit register to start transmission
	nop
	nop
	btfss	PIR1,TXIF			; poll for transmit buffer to empty (can't load next byte until then)
	goto	$-3					; TXIF goes high once byte transfers from TXREG to transmit shift register
	movlw	0x61				;"a"
	movwf	TXREG				; move byte to transmit register to start transmission
	nop
	nop
	btfss	PIR1,TXIF			; poll for transmit buffer to empty (can't load next byte until then)
	goto	$-3					; TXIF goes high once byte transfers from TXREG to transmit shift register
	movlw	0x69				;"i"
	movwf	TXREG				; move byte to transmit register to start transmission
	nop
	nop
	btfss	PIR1,TXIF			; poll for transmit buffer to empty (can't load next byte until then)
	goto	$-3					; TXIF goes high once byte transfers from TXREG to transmit shift register
	movlw	0x6e				;"n"
	movwf	TXREG				; move byte to transmit register to start transmission
	nop
	nop
	btfss	PIR1,TXIF			; poll for transmit buffer to empty (can't load next byte until then)
	goto	$-3					; TXIF goes high once byte transfers from TXREG to transmit shift register
	movlw	COMMA				;","
	movwf	TXREG				; move byte to transmit register to start transmission
	nop
	nop
	btfss	PIR1,TXIF			; poll for transmit buffer to empty (can't load next byte until then)
	goto	$-3					; TXIF goes high once byte transfers from TXREG to transmit shift register
	movlw	0x74				;"t"
	movwf	TXREG				; move byte to transmit register to start transmission
	nop
	nop
	btfss	PIR1,TXIF			; poll for transmit buffer to empty (can't load next byte until then)
	goto	$-3					; TXIF goes high once byte transfers from TXREG to transmit shift register
	movlw	0x65				;"e"
	movwf	TXREG				; move byte to transmit register to start transmission
	nop
	nop
	btfss	PIR1,TXIF			; poll for transmit buffer to empty (can't load next byte until then)
	goto	$-3					; TXIF goes high once byte transfers from TXREG to transmit shift register
	movlw	0x6d				;"m"
	movwf	TXREG				; move byte to transmit register to start transmission
	nop
	nop
	btfss	PIR1,TXIF			; poll for transmit buffer to empty (can't load next byte until then)
	goto	$-3					; TXIF goes high once byte transfers from TXREG to transmit shift register
	movlw	0x70				;"p"
	movwf	TXREG				; move byte to transmit register to start transmission
	nop
	nop
	btfss	PIR1,TXIF			; poll for transmit buffer to empty (can't load next byte until then)
	goto	$-3					; TXIF goes high once byte transfers from TXREG to transmit shift register
	movlw	CR					;<carriage return>
	movwf	TXREG				; move byte to transmit register to start transmission
	nop
	nop
	btfss	PIR1,TXIF			; poll for transmit buffer to empty (can't load next byte until then)
	goto	$-3					; TXIF goes high once byte transfers from TXREG to transmit shift register
	movlw	LF					;<line feed>
	movwf	TXREG				; move byte to transmit register to start transmission
	nop
	nop
	btfss	PIR1,TXIF			; poll for transmit buffer to empty (can't load next byte until then)
	goto	$-3					; TXIF goes high once byte transfers from TXREG to transmit shift register
	return



;***************************************************************
tx_nul		; Routine to transmit the text "nul,<carriage return><line feed>"
	banksel	PORTA				; ensure in page 0
	movlw	0x6e				;"n"
	movwf	TXREG				; move byte to transmit register to start transmission
	nop
	nop
	btfss	PIR1,TXIF			; poll for transmit buffer to empty (can't load next byte until then)
	goto	$-3					; TXIF goes high once byte transfers from TXREG to transmit shift register
	movlw	0x75				;"u"
	movwf	TXREG				; move byte to transmit register to start transmission
	nop
	nop
	btfss	PIR1,TXIF			; poll for transmit buffer to empty (can't load next byte until then)
	goto	$-3					; TXIF goes high once byte transfers from TXREG to transmit shift register
	movlw	0x6c				;"l"
	movwf	TXREG				; move byte to transmit register to start transmission
	nop
	nop
	btfss	PIR1,TXIF			; poll for transmit buffer to empty (can't load next byte until then)
	goto	$-3					; TXIF goes high once byte transfers from TXREG to transmit shift register
	movlw	COMMA				;","
	movwf	TXREG				; move byte to transmit register to start transmission
	nop
	nop
	btfss	PIR1,TXIF			; poll for transmit buffer to empty (can't load next byte until then)
	goto	$-3					; TXIF goes high once byte transfers from TXREG to transmit shift register
	movlw	CR					;<carriage return>
	movwf	TXREG				; move byte to transmit register to start transmission
	nop
	nop
	btfss	PIR1,TXIF			; poll for transmit buffer to empty (can't load next byte until then)
	goto	$-3					; TXIF goes high once byte transfers from TXREG to transmit shift register
	movlw	LF					;<line feed>
	movwf	TXREG				; move byte to transmit register to start transmission
	nop
	nop
	btfss	PIR1,TXIF			; poll for transmit buffer to empty (can't load next byte until then)
	goto	$-3					; TXIF goes high once byte transfers from TXREG to transmit shift register
	return



;***************************************************************
tx_brk		; Routine to transmit the text "BREAK,<carriage return><line feed>"
	banksel	PORTA				; ensure in page 0
	movlw	0x42				;"B"
	movwf	TXREG				; move byte to transmit register to start transmission
	nop
	nop
	btfss	PIR1,TXIF			; poll for transmit buffer to empty (can't load next byte until then)
	goto	$-3					; TXIF goes high once byte transfers from TXREG to transmit shift register
	movlw	0x52				;"R"
	movwf	TXREG				; move byte to transmit register to start transmission
	nop
	nop
	btfss	PIR1,TXIF			; poll for transmit buffer to empty (can't load next byte until then)
	goto	$-3					; TXIF goes high once byte transfers from TXREG to transmit shift register
	movlw	0x45				;"E"
	movwf	TXREG				; move byte to transmit register to start transmission
	nop
	nop
	btfss	PIR1,TXIF			; poll for transmit buffer to empty (can't load next byte until then)
	goto	$-3					; TXIF goes high once byte transfers from TXREG to transmit shift register
	movlw	0x41				;"A"
	movwf	TXREG				; move byte to transmit register to start transmission
	nop
	nop
	btfss	PIR1,TXIF			; poll for transmit buffer to empty (can't load next byte until then)
	goto	$-3					; TXIF goes high once byte transfers from TXREG to transmit shift register
	movlw	0x4B				;"K"
	movwf	TXREG				; move byte to transmit register to start transmission
	nop
	nop
	btfss	PIR1,TXIF			; poll for transmit buffer to empty (can't load next byte until then)
	goto	$-3					; TXIF goes high once byte transfers from TXREG to transmit shift register
	movlw	COMMA				;","
	movwf	TXREG				; move byte to transmit register to start transmission
	nop
	nop
	btfss	PIR1,TXIF			; poll for transmit buffer to empty (can't load next byte until then)
	goto	$-3					; TXIF goes high once byte transfers from TXREG to transmit shift register
	movlw	CR					;<carriage return>
	movwf	TXREG				; move byte to transmit register to start transmission
	nop
	nop
	btfss	PIR1,TXIF			; poll for transmit buffer to empty (can't load next byte until then)
	goto	$-3					; TXIF goes high once byte transfers from TXREG to transmit shift register
	movlw	LF					;<line feed>
	movwf	TXREG				; move byte to transmit register to start transmission
	nop
	nop
	btfss	PIR1,TXIF			; poll for transmit buffer to empty (can't load next byte until then)
	goto	$-3					; TXIF goes high once byte transfers from TXREG to transmit shift register
	return
	
	


	END                       ; directive 'end of program'

